const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const redis = require('../../plugin/DataBase/Redis')
const simpleGit = require('simple-git')
const fs = require("fs")
const path = require("path")
const { isBinaryFileSync } = require('isbinaryfile')

class gitCodeStats extends API {
    constructor() {
        super()

        this.setMethod("GET")
        this.setPath("/Repos/CodeStats")
    }

    // 定义支持的语言后缀
    languageExtensions = {
        "JavaScript": [".js", ".jsx", ".mjs", ".cjs"],
        "TypeScript": [".ts", ".tsx"],
        "Python": [".py"],
        "Java": [".java"],
        "C": [".c"],
        "C++": [".cpp", ".cc", ".cxx", ".h", ".hpp"],
        "C#": [".cs"],
        "Go": [".go"],
        "PHP": [".php"],
        "HTML": [".html", ".htm"],
        "CSS": [".css"],
        "SCSS": [".scss", ".sass"],
        "JSON": [".json"],
        "YAML": [".yml", ".yaml"],
        "Markdown": [".md"],
        "Shell": [".sh"],
        "Bash": [".bash"],
        "PowerShell": [".ps1"],
        "Ruby": [".rb"],
        "Vue": [".vue"],  // Vue文件
        "LESS": [".less"],  // LESS样式文件
        "Sass": [".sass"],  // Sass文件
        "Haml": [".haml"],  // Haml文件
        "Elixir": [".ex", ".exs"],  // Elixir语言文件
        "Erlang": [".erl", ".hrl"],  // Erlang语言文件
        "Swift": [".swift"],  // Swift语言文件
        "Kotlin": [".kt", ".kts"],  // Kotlin语言文件
        "Rust": [".rs"],  // Rust语言文件
        "Dart": [".dart"],  // Dart语言文件
        "Lua": [".lua"],  // Lua语言文件
        "R": [".r"],  // R语言文件
        "Perl": [".pl"],  // Perl语言文件
        "SQL": [".sql"],  // SQL文件
        "Dockerfile": [".dockerfile", "Dockerfile"],  // Dockerfile
        "GraphQL": [".graphql", ".gql"],  // GraphQL文件
        "JSON5": [".json5"],  // JSON5文件
        "TOML": [".toml"],  // TOML文件
        "Text": [".txt", ".text"],  // 文本文件
        "Assembly": [".asm", ".s"],
        "Objective-C": [".m", ".mm"],
        "F#": [".fs", ".fsi", ".fsx", ".fsscript"],
        "Haskell": [".hs", ".lhs"],
        "OCaml": [".ml", ".mli"],
        "Nim": [".nim", ".nims"],
        "Julia": [".jl"],
        "Fortran": [".f", ".for", ".f90", ".f95"],
        "COBOL": [".cbl", ".cob"],
        "VHDL": [".vhd", ".vhdl"],
        "Verilog": [".v", ".vh"],
        "Ada": [".adb", ".ads"],
        "Matlab": [".m"],  // 注意 Objective-C 也使用 .m，可根据上下文判断
        "Scala": [".scala"],
        "VB.NET": [".vb"],
        "Groovy": [".groovy", ".gvy", ".gy", ".gsh"],
        "Makefile": ["Makefile", "makefile", ".mk"],
        "Clojure": [".clj", ".cljs", ".cljc", ".edn"],
        "Common Lisp": [".lisp", ".lsp"],
        "Scheme": [".scm", ".ss"],
        "Prolog": [".pl", ".pro", ".P"],
        "Smalltalk": [".st", ".squeak"],
        "Tcl": [".tcl"],
        "Crystal": [".cr"],
        "Solidity": [".sol"],
        "Vim script": [".vim"],
        "ReScript": [".res", ".resi"],
        "ReasonML": [".re", ".rei"],
        "Pug": [".pug", ".jade"],
        "Handlebars": [".hbs", ".handlebars"],
        "XML": [".xml"],
        "INI": [".ini", ".cfg", ".conf"],
        "Log": [".log"],
        "ENV": [".env"],
        "ReStructuredText": [".rst"],
        "AsciiDoc": [".adoc", ".asciidoc"],
        "Racket": [".rkt"],
        "Zig": [".zig"],
        "Haxe": [".hx"],
        "Dotenv": [".env"],
        "Config": [".config"],
        "PlantUML": [".puml", ".plantuml"],
        "Visual Basic": [".bas", ".vbs"],
        "Batch": [".bat", ".cmd"],
        "BibTeX": [".bib"],
        "TeX/LaTeX": [".tex", ".ltx", ".sty", ".cls"],
        "Apache": [".htaccess", "httpd.conf"],
        "NGINX": ["nginx.conf"],
        "Terraform": [".tf", ".tfvars"],
        "HCL": [".hcl"],
        "QML": [".qml"],
        "Cue": [".cue"],
        "GDScript": [".gd"], // 用于 Godot 引擎
        "ANTLR": [".g4"],
        "Pascal": [".pas"],
        "Logtalk": [".lgt"],
        "Awk": [".awk"],
        "Sed": [".sed"],
        "ConfigScript": [".conf", ".cfg"],
        "YANG": [".yang"],
        "NetLogo": [".nlogo"],
        "Other": [".bin", ".dat", ".exe", ".dll", ".obj", ".so", ".class"] // 二进制及无法识别的文件
    }

    // 判断文件是否为二进制文件
    isBinaryFile(filePath) {
        try {
            return isBinaryFileSync(filePath)
        } catch (err) {
            return false
        }
    }

    // 统计单个文件的行数
    countLines(filePath) {
        return new Promise((resolve) => {
            if (this.isBinaryFile(filePath)) {
                return resolve(0) // 二进制文件直接跳过
            }
            let lineCount = 0;
            const stream = fs.createReadStream(filePath, 'utf8')
            stream.on("data", (chunk) => {
                lineCount += chunk.toString().split("\n").length
            });
            stream.on("end", () => resolve(lineCount))
            stream.on("error", () => resolve(0)) // 读取错误默认返回 0
        })
    }

    async onRequest(req, res) {
        let { uuid, session, id } = req.query

        if ([uuid, session, id].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT state, path FROM repos WHERE create_user = ? AND id = ?'

        let r = await db.query(sql, [uuid, id])
        if (!r || r.length === 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '未找到仓库'
            })

        if (r[0].state !== 1 || !r[0].path)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库未成功克隆！'
            })

        const cachedCodeStats = await redis.get(`codeStats:${r[0].path}`)
        if(cachedCodeStats)
            return res.json({
                ...BaseStdResponse.OK,
                data: JSON.parse(cachedCodeStats)
            })

        let totalLines = 0
        let languageStats = {}
        try {
            const git = simpleGit(r[0].path)
            const files = await git.raw(["ls-files"]) // 获取所有 Git 跟踪的文件
            const fileList = files.split("\n").map(f => f.trim()).filter(f => f)

            for (const file of fileList) {
                const fullPath = path.join(r[0].path, file)
                const ext = path.extname(file)
                const language = Object.keys(this.languageExtensions).find(lang =>
                    this.languageExtensions[lang].includes(ext)
                )

                if (language) {
                    const lines = await this.countLines(fullPath)
                    totalLines += lines
                    languageStats[language] = (languageStats[language] || 0) + lines
                } else {
                    // 如果文件不属于任何已知语言，归类为其他
                    const lines = await this.countLines(fullPath)
                    totalLines += lines
                    languageStats["Other"] = (languageStats["Other"] || 0) + lines
                }
            }
        } catch (error) {
            this.logger.error('获取仓库代码信息失败！' + error.stack)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '获取仓库代码信息失败！'
            })
        }

        const data = { totalLines, languageStats }

        res.json({
            ...BaseStdResponse.OK,
            data
        })
        redis.set(`codeStats:${r[0].path}`, JSON.stringify(data), {
            EX: 172800
        })
    }
}

module.exports.gitCodeStats = gitCodeStats
