const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const simpleGit = require('simple-git')

class GetRepoStatus extends API {
    constructor() {
        super()

        this.setMethod("GET")
        this.setPath("/Repos/Status")
    }

    async onRequest(req, res) {
        let { uuid, session, id } = req.query

        if ([uuid, session, id].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT name, logo, create_time, update_time, state, path, url FROM repos WHERE create_user = ? AND id = ?'
        let r = await db.query(sql, [uuid, id])
        if (!r || r.length === 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '未找到仓库'
            })

        if (r[0].state !== 1 || !r[0].path)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库未成功克隆！'
            })

        try {
            const git = simpleGit()
            await git.cwd(r[0].path)
            let status = await git.status()
            const branches = await git.branch()
            const remoteBranches = Object.keys(branches.branches)
                .filter(branch => branch.startsWith('remotes/origin/'))
                .reduce((acc, branch) => {
                    const newBranchName = branch.replace('remotes/origin/', '')
                    acc[newBranchName] = {
                        ...branches.branches[branch], 
                        name: newBranchName
                    };
                    return acc
                }, {})

            res.json({
                ...BaseStdResponse.OK,
                data: {
                    status,
                    branches: remoteBranches,
                    info: {
                        name: r[0].name,
                        url: r[0].url,
                        logo: r[0].logo,
                        create_time: r[0].create_time,
                        update_time: r[0].update_time,
                        state: r[0].state
                    },
                }
            })

        } catch (error) {
            this.logger.error('获取仓库状态失败！' + error.stack)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '获取仓库状态失败！'
            })
        }

    }
}

module.exports.GetRepoStatus = GetRepoStatus
