const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const redis = require('../../plugin/DataBase/Redis')
const simpleGit = require('simple-git')

class GetRepoLog extends API {
    constructor() {
        super()

        this.setMethod("GET")
        this.setPath("/Repos/Log")
    }

    async onRequest(req, res) {
        let { uuid, session, id } = req.query

        if ([uuid, session, id].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT state, path, url FROM repos WHERE create_user = ? AND id = ?'
        let r = await db.query(sql, [uuid, id])
        if (!r || r.length === 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '未找到仓库'
            })

        if (r[0].state !== 1 || !r[0].path)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库未成功克隆！'
            })

        try {
            let rawLogs
            const redisKey = `gitLogs:${r[0].path}`
            const cachedLogs = await redis.get(redisKey)

            if (cachedLogs)
                rawLogs = JSON.parse(cachedLogs)
            else {
                const git = simpleGit()
                await git.cwd(r[0].path)

                rawLogs = await git.raw([
                    "log",
                    "--no-merges",
                    "--pretty=format:%H|%an|%ae|%ad|%s",
                    "--shortstat"
                ])

                redis.set(redisKey, JSON.stringify(rawLogs), {
                    EX: 172800
                })
            }

            const lines = rawLogs.split("\n")
            const commits = []
            let current = null

            for (let i = 0; i < lines.length; i++) {
                const line = lines[i].trim()

                // 提交信息行
                if (line.includes("|") && !line.includes("file changed")) {
                    const [hash, name, email, date, ...messageParts] = line.split("|")

                    current = {
                        hash,
                        name,
                        email,
                        date,
                        message: messageParts.join("|").trim()
                    }

                    commits.push(current)
                }
            }

            res.json({
                ...BaseStdResponse.OK,
                data: commits
            })
        } catch (err) {
            res.json({
                ...BaseStdResponse.ERR,
                message: "Git 日志分析失败"
            });
        }
    }
}

module.exports.GetRepoLog = GetRepoLog
