const API = require("../../lib/API")
const db = require("../../plugin/DataBase/db")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const axios = require("axios")
const { core_url } = require('../../config.json')

class GetRepoList extends API {
    constructor() {
        super()

        this.setPath('/Repos/List')
        this.setMethod('GET')
    }

    async onRequest(req, res) {
        let {
            uuid,
            session,
            keyword
        } = req.query

        if ([uuid, session].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT id, name, url, create_time, state, logo, update_time FROM repos WHERE create_user = ?'
        let params = [uuid]

        if ((keyword ?? '').trim() !== '') {
            sql += ' AND name LIKE ?'
            params.push(`%${keyword}%`)
        }

        sql += ' ORDER BY update_time DESC'

        let rows = await db.query(sql, params)

        if (!rows || rows.length === 0)
            return res.json({
                ...BaseStdResponse.OK,
                data: []
            })

        const status_url = core_url + '/git/status'

        try {
            // 并行执行所有请求
            const statusPromises = rows.map(item =>
                axios.post(status_url, { uuid, repo_url: item.url, repo_id: item.id })
                    .then(res => ({ ...item, status: res.data }))
                    .catch(() => ({ ...item, status: null }))
            )

            const items = await Promise.all(statusPromises)

            res.json({
                ...BaseStdResponse.OK,
                data: items
            })
        } catch (error) {
            res.json({
                ...BaseStdResponse.ERR,
                message: '获取仓库状态失败'
            })
        }
    }
}

module.exports.GetRepoList = GetRepoList