const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const simpleGit = require('simple-git')

class GetFiles extends API {
    constructor() {
        super()

        this.setMethod("GET")
        this.setPath("/Repos/GetFiles")
    }

    buildTreeStructure(paths) {
        const root = []

        paths.forEach(filePath => {
            const parts = filePath.split('/')
            let currentLevel = root

            parts.forEach((part, index) => {
                const existingNode = currentLevel.find(node => node.title === part)

                // 如果存在节点，则继续深入
                if (existingNode) {
                    currentLevel = existingNode.children
                } else {
                    // 判断是否为文件
                    const isFile = index === parts.length - 1

                    // 新增节点
                    const newNode = {
                        title: part,
                        key: filePath,
                        type: isFile ? 'file' : 'folder',
                        children: []
                    }

                    // 判断是否为叶子节点
                    if (index === parts.length - 1) {
                        newNode.icon = () => h(IconDriveFile)
                        newNode.children = undefined
                    }

                    currentLevel.push(newNode)
                    currentLevel = newNode.children
                }
            })
        })

        return root
    }

    async onRequest(req, res) {
        let { uuid, session, id } = req.query

        if ([uuid, session, id].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT name, logo, create_time, update_time, state, path, url FROM repos WHERE create_user = ? AND id = ?'
        let r = await db.query(sql, [uuid, id])
        if (!r || r.length === 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '未找到仓库'
            })

        if (r[0].state !== 1 || !r[0].path)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库未成功克隆！'
            })

        try {
            const git = simpleGit()
            await git.cwd(r[0].path)
            const treeResult = await git.raw(['ls-tree', '-r', '--name-only', 'HEAD'])

            const filePaths = treeResult.split('\n').filter(path => path) // 去掉空行

            // 构建树结构
            const treeStructure = this.buildTreeStructure(filePaths)

            res.json({
                ...BaseStdResponse.OK,
                data: treeStructure
            })

        } catch (error) {
            this.logger.error('获取文件列表失败！' + error.stack)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '获取文件列表失败！'
            })
        }

    }
}

module.exports.GetFiles = GetFiles
