const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const simpleGit = require('simple-git')

class GetCommitDetail extends API {
    constructor() {
        super()
        this.setMethod("GET")
        this.setPath("/Repos/GetCommitDetail")
    }

    async onRequest(req, res) {
        let { uuid, session, id, hash } = req.query

        if ([uuid, session, id, hash].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        let sql = 'SELECT state, path FROM repos WHERE create_user = ? AND id = ?'
        let r = await db.query(sql, [uuid, id])
        if (!r || r.length === 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '未找到仓库'
            })

        if (r[0].state !== 1 || !r[0].path)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库未成功克隆！'
            })

        try {
            const git = simpleGit()
            await git.cwd(r[0].path)

            const showCommitDetails = (hash) => {
                return new Promise((resolve, reject) => {
                    git.show([hash], (err, result) => {
                        if (err) reject(err)
                        resolve(result)
                    })
                })
            }

            const result = await showCommitDetails(hash)

            // 更新正则表达式，捕获多行 commit message（包括 body）
            const commitRegex = /^commit (\w+)\nAuthor: (.+) <(.+)>\nDate:\s+(.+)\n\n([\s\S]+?)(?=\ndiff --git|$)/
            const match = result.match(commitRegex)
            if (!match)
                return res.json({
                    ...BaseStdResponse.ERR,
                    msg: '解析提交详情失败！'
                })

            // 拆分 message 和 body
            const fullMessage = match[5].trim().split("\n")
            const message = fullMessage[0] // 第一行短描述
            const body = fullMessage.slice(1).join("\n").trim() // 多行 body

            const commitInfo = {
                hash: match[1],
                author: match[2],
                email: match[3],
                date: match[4],
                message,
                body,
                files: [],
                diffs: []
            }

            // 检查是否为合并提交
            const isMergeCommit = message.startsWith('Merge')

            if (isMergeCommit) {
                // 合并提交处理：如果是合并提交，直接返回合并信息
                commitInfo.mergeInfo = {
                    message: message,  // 这里返回合并提交的信息
                    body: body         // 可能包含合并的详情
                }
            } else {
                // 解析 diff 信息
                const diffRegex = /diff --git a\/(.+?) b\/.+?\nindex .+\n--- a\/.+?\n\+\+\+ b\/.+?\n([\s\S]+?)(?=(diff --git a\/|$))/g
                let diffMatch
                while ((diffMatch = diffRegex.exec(result)) !== null) {
                    commitInfo.files.push(diffMatch[1])
                    commitInfo.diffs.push({
                        file: diffMatch[1],
                        changes: diffMatch[2].trim().split('\n')
                    })
                }
            }

            res.json({
                ...BaseStdResponse.OK,
                data: commitInfo
            })
        } catch (error) {
            this.logger.error('获取提交详情失败！' + error.stack)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '获取提交详情失败！'
            })
        }
    }
}

module.exports.GetCommitDetail = GetCommitDetail
