const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const axios = require('axios')
const { core_url } = require('../../config.json')

// 添加仓库
class AddRepo extends API {
    constructor() {
        super()

        this.setMethod("POST")
        this.setPath("/Repos")
    }

    checkName(name) {
        const regex = /^(?!\.)(?!.*\/{2,})(?!.*\.git$)[\u4E00-\u9FA5A-Za-z0-9._-]+(?<!\/)$/
        return regex.test(name)
    }

    async onRequest(req, res) {
        let { uuid, session, name, url } = req.body

        if ([uuid, session, name, url].some(value => value === '' || value === null || value === undefined))
            return res.json({
                ...BaseStdResponse.MISSING_PARAMETER
            })

        if (!this.checkName(name))
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '仓库名称不合法！'
            })

        // 检查 session
        if (!await AccessControl.checkSession(uuid, session))
            return res.status(401).json({
                ...BaseStdResponse.ACCESS_DENIED
            })

        // 检查仓库是否重复
        let sql = 'SELECT id FROM repos WHERE create_user = ? AND url = ?'
        let r = await db.query(sql, [uuid, url])
        if (r && r.length > 0)
            return res.json({
                ...BaseStdResponse.ERR,
                msg: '用户已添加该仓库！'
            })

        const time = new Date().getTime()
        sql = 'INSERT INTO repos (name, url, create_user, create_time) VALUES (?, ?, ?, ?)'

        let result = await db.query(sql, [name, url, uuid, time])

        if (result && result.affectedRows > 0) {
            res.json({
                ...BaseStdResponse.OK
            })

            // 发起克隆
            try {
                const clone_url = core_url + '/git/clone'
                await axios.post(clone_url, { uuid, repo_url: url, repo_id: result.insertId })
            } catch (error) {
                this.logger.error('克隆仓库失败！' + error.stack)
            }
        } else {
            res.json({ ...BaseStdResponse.ERR, endpoint: 7894378, msg: '添加仓库失败！' })
        }
    }
}

module.exports.AddRepo = AddRepo
