const API = require("../../lib/API")
const AccessControl = require("../../lib/AccessControl")
const { BaseStdResponse } = require("../../BaseStdResponse")
const db = require("../../plugin/DataBase/db")
const Redis = require('../../plugin/DataBase/Redis')
const { v4: uuidv4 } = require('uuid')
const axios = require("axios")
const { core_url } = require("../../config.json")

class AIChat extends API {
    constructor() {
        super()
        this.setMethod("POST")
        this.setPath("/AI/Chat")
    }

    async onRequest(req, res) {
        const { uuid, session, message } = req.body

        // 参数校验
        if (!uuid || !session || !message)
            return res.json({ ...BaseStdResponse.MISSING_PARAMETER })


        // session 校验
        const sessionValid = await AccessControl.checkSession(uuid, session)
        if (!sessionValid) {
            return res.status(401).json({ ...BaseStdResponse.ACCESS_DENIED })
        }

        const userTime = new Date().getTime() // 用户消息发送时间
        const msgid = uuidv4()

        res.json({
            ...BaseStdResponse.OK,
            id: msgid
        })

        try {
            const endpoint = `${core_url}/ai/chat`
            const aiResponse = await axios.post(endpoint, { uuid, message })

            const aiData = aiResponse?.data?.msg
            if (!aiData)
                throw new Error("AI响应格式错误")

            const aiTime = new Date().getTime()

            await Redis.set(`message:${msgid}`, aiData, {
                EX: 600
            })

            // 异步写入数据库
            const insertUserMessage = db.query(
                'INSERT INTO messages (uuid, content, time, type) VALUES (?, ?, ?, "user")',
                [uuid, message, userTime]
            )

            const insertAIMessage = db.query(
                'INSERT INTO messages (uuid, content, time, type) VALUES (?, ?, ?, "ai")',
                [uuid, aiData, aiTime]
            )

            await Promise.all([insertUserMessage, insertAIMessage])
        } catch (error) {
            this.logger.error("消息发送失败：" + error.stack)
            await Redis.set(`message:${msgid}`, '消息发送失败,请稍后再试', {
                EX: 600
            })
        }
    }
}

module.exports.AIChat = AIChat
